/**
 * QuikForms External Verification Plugin
 * A hybrid plugin that verifies form field values against external APIs
 * on field blur. The JavaScript triggers the callout via QuikFormsPlugins.callout(),
 * while Apex (IQuikFormsCalloutHandler) handles the actual HTTP request.
 *
 * Configuration (via plugin config):
 *   fields - Array of field configs: [{ fieldId, verificationType, objectField }]
 *     fieldId           - Exact field ID to match
 *     objectField       - Object.Field pattern to match (e.g., "Contact.Email")
 *     verificationType  - Type of verification (e.g., "email", "phone", "domain")
 *
 * @version 1.0.0
 * @license MIT
 */
(function() {
  'use strict';

  /* ------------------------------------------------------------------ */
  /*  Constants                                                         */
  /* ------------------------------------------------------------------ */

  var PLUGIN_NAME = 'externalVerification';

  var CSS_PREFIX = 'qfp-verify';

  var INDICATOR_CLASSES = {
    container: CSS_PREFIX + '-indicator',
    success:   CSS_PREFIX + '-success',
    pending:   CSS_PREFIX + '-pending',
    warning:   CSS_PREFIX + '-warning'
  };

  /* ------------------------------------------------------------------ */
  /*  Inline styles (injected once)                                     */
  /* ------------------------------------------------------------------ */

  var stylesInjected = false;

  /**
   * Inject scoped CSS for verification indicators.
   * Uses CSS custom properties from the QuikForms theme where applicable.
   */
  function injectStyles() {
    if (stylesInjected) {
      return;
    }
    stylesInjected = true;

    var css =
      '.' + INDICATOR_CLASSES.container + ' {' +
        'display: inline-flex;' +
        'align-items: center;' +
        'gap: 4px;' +
        'font-size: 12px;' +
        'font-weight: 500;' +
        'line-height: 1.4;' +
        'padding: 2px 8px;' +
        'border-radius: var(--qf-radius-sm, 4px);' +
        'margin-top: 4px;' +
        'transition: opacity 0.2s ease;' +
      '}' +

      '.' + INDICATOR_CLASSES.success + ' {' +
        'color: var(--qf-success-text, #166534);' +
        'background: var(--qf-success-bg, rgba(22, 163, 74, 0.1));' +
        'border: 1px solid var(--qf-success-border, rgba(22, 163, 74, 0.3));' +
      '}' +

      '.' + INDICATOR_CLASSES.pending + ' {' +
        'color: var(--qf-info-text, #1e40af);' +
        'background: var(--qf-info-bg, rgba(59, 130, 246, 0.1));' +
        'border: 1px solid var(--qf-info-border, rgba(59, 130, 246, 0.3));' +
      '}' +

      '.' + INDICATOR_CLASSES.pending + ' .qfp-verify-spinner {' +
        'display: inline-block;' +
        'width: 12px;' +
        'height: 12px;' +
        'border: 2px solid var(--qf-info-border, rgba(59, 130, 246, 0.3));' +
        'border-top-color: var(--qf-info-text, #1e40af);' +
        'border-radius: 50%;' +
        'animation: qfp-verify-spin 0.8s linear infinite;' +
      '}' +

      '.' + INDICATOR_CLASSES.warning + ' {' +
        'color: var(--qf-warning-text, #854d0e);' +
        'background: var(--qf-warning-bg, rgba(234, 179, 8, 0.1));' +
        'border: 1px solid var(--qf-warning-border, rgba(234, 179, 8, 0.3));' +
      '}' +

      '@keyframes qfp-verify-spin {' +
        'to { transform: rotate(360deg); }' +
      '}';

    var styleEl = document.createElement('style');
    styleEl.setAttribute('data-qfp-verify', 'true');
    styleEl.textContent = css;
    document.head.appendChild(styleEl);
  }

  /* ------------------------------------------------------------------ */
  /*  State                                                             */
  /* ------------------------------------------------------------------ */

  /**
   * Plugin configuration, set during onFormLoad.
   * @type {{ fields: Array<{ fieldId: string, objectField: string, verificationType: string }> }}
   */
  var pluginConfig = null;

  /**
   * Tracks in-flight verification requests per field to support cancellation.
   * @type {Object<string, number>}
   */
  var pendingRequests = {};

  /* ------------------------------------------------------------------ */
  /*  Helpers                                                           */
  /* ------------------------------------------------------------------ */

  /**
   * Determine if a field is configured for verification.
   * Matches by exact fieldId or by objectField pattern.
   *
   * @param {string} fieldId - The field's unique identifier
   * @param {object} field   - The field definition from QuikForms
   * @returns {object|null} The matching field config, or null if not configured
   */
  function getFieldConfig(fieldId, field) {
    if (!pluginConfig || !pluginConfig.fields || !Array.isArray(pluginConfig.fields)) {
      return null;
    }

    for (var i = 0; i < pluginConfig.fields.length; i++) {
      var fc = pluginConfig.fields[i];

      // Match by exact fieldId
      if (fc.fieldId && fc.fieldId === fieldId) {
        return fc;
      }

      // Match by objectField pattern (e.g., "Contact.Email")
      if (fc.objectField && field && field.objectField === fc.objectField) {
        return fc;
      }
    }

    return null;
  }

  /**
   * Find the DOM container element for a field by its ID.
   * Looks for the standard QuikForms field wrapper structure.
   *
   * @param {string} fieldId
   * @returns {HTMLElement|null}
   */
  function getFieldContainer(fieldId) {
    // Try the standard QuikForms field container first
    var container = document.querySelector('[data-field-id="' + fieldId + '"]');
    if (container) {
      return container;
    }

    // Fallback: find the input and use its parent
    var input = document.getElementById('field-' + fieldId);
    if (input && input.parentElement) {
      return input.parentElement;
    }

    return null;
  }

  /**
   * Remove any existing verification indicator from a field container.
   *
   * @param {string} fieldId
   */
  function clearIndicator(fieldId) {
    var container = getFieldContainer(fieldId);
    if (!container) {
      return;
    }

    var existing = container.querySelectorAll('.' + INDICATOR_CLASSES.container);
    for (var i = 0; i < existing.length; i++) {
      existing[i].parentNode.removeChild(existing[i]);
    }
  }

  /**
   * Show a verification indicator next to a field.
   *
   * @param {string} fieldId
   * @param {'success'|'pending'|'warning'} state - The indicator state
   * @param {string} message - The message text to display
   */
  function showIndicator(fieldId, state, message) {
    clearIndicator(fieldId);

    var container = getFieldContainer(fieldId);
    if (!container) {
      return;
    }

    var span = document.createElement('span');
    span.className = INDICATOR_CLASSES.container + ' ' + INDICATOR_CLASSES[state];

    if (state === 'pending') {
      span.innerHTML = '<span class="qfp-verify-spinner"></span> ' + escapeHtml(message);
    } else if (state === 'success') {
      span.innerHTML = '<span aria-hidden="true">&#10003;</span> ' + escapeHtml(message);
    } else if (state === 'warning') {
      span.innerHTML = '<span aria-hidden="true">&#9888;</span> ' + escapeHtml(message);
    }

    span.setAttribute('role', 'status');
    span.setAttribute('aria-live', 'polite');

    container.appendChild(span);
  }

  /**
   * Escape a string for safe insertion into HTML text content.
   *
   * @param {string} str
   * @returns {string}
   */
  function escapeHtml(str) {
    return String(str)
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;');
  }

  /* ------------------------------------------------------------------ */
  /*  Hook: onFieldBlur                                                 */
  /* ------------------------------------------------------------------ */

  /**
   * onFieldBlur hook: Triggered when a configured field loses focus.
   * Initiates an external verification callout for the field value.
   *
   * @param {object} context - { fieldId, field, value, formData, locale, formConfigId }
   * @returns {null}
   */
  function onFieldBlur(context) {
    if (!context || !context.fieldId) {
      return null;
    }

    var fieldConfig = getFieldConfig(context.fieldId, context.field);
    if (!fieldConfig) {
      return null;
    }

    var value = context.value;

    // Skip verification for empty/blank values
    if (value === null || value === undefined || String(value).trim() === '') {
      clearIndicator(context.fieldId);
      return null;
    }

    // Cancel any in-flight request for this field
    var requestId = ++pendingRequests[context.fieldId] || 1;
    pendingRequests[context.fieldId] = requestId;

    // Show pending indicator
    showIndicator(context.fieldId, 'pending', 'Verifying...');

    // Resolve formConfigId from context
    var formConfigId = context.formConfigId || (context.formData && context.formData.formConfigId) || '';

    // Execute callout via the plugin framework
    var calloutPromise;
    try {
      calloutPromise = QuikFormsPlugins.callout({
        pluginName: PLUGIN_NAME,
        params: {
          fieldId: context.fieldId,
          value: String(value),
          verificationType: fieldConfig.verificationType || 'default'
        },
        fc: formConfigId
      });
    } catch (e) {
      // If callout() throws synchronously (e.g., missing QuikFormsPlugins)
      showIndicator(context.fieldId, 'warning', 'Could not verify');
      console.warn('QuikForms ExternalVerification: callout failed synchronously', e);
      return null;
    }

    // Handle the async response
    calloutPromise
      .then(function(response) {
        // Check if this request is still the latest for this field
        if (pendingRequests[context.fieldId] !== requestId) {
          return; // Stale request, ignore
        }

        // Parse the response to determine verification result
        var verified = false;
        var message = 'Verified';

        if (response && typeof response === 'object') {
          if (response.verified === true || response.valid === true || response.success === true) {
            verified = true;
          }
          if (response.message) {
            message = String(response.message);
          }
        } else if (response === true) {
          verified = true;
        }

        if (verified) {
          showIndicator(context.fieldId, 'success', message);
        } else {
          showIndicator(context.fieldId, 'warning', message || 'Could not verify');
        }
      })
      .catch(function(error) {
        // Check if this request is still the latest for this field
        if (pendingRequests[context.fieldId] !== requestId) {
          return; // Stale request, ignore
        }

        var errorMessage = 'Could not verify';
        if (error && error.message) {
          errorMessage = error.message;
        }

        showIndicator(context.fieldId, 'warning', errorMessage);
        console.warn('QuikForms ExternalVerification: callout error for field ' + context.fieldId, error);
      });

    return null;
  }

  /* ------------------------------------------------------------------ */
  /*  Hook: onFormLoad                                                  */
  /* ------------------------------------------------------------------ */

  /**
   * onFormLoad hook: Initializes the plugin with the form configuration.
   * Extracts the plugin config to determine which fields to verify.
   *
   * @param {object} context - { formData, locale, plugins }
   * @returns {null}
   */
  function onFormLoad(context) {
    injectStyles();

    // Extract plugin configuration from the context
    if (context && context.plugins && context.plugins[PLUGIN_NAME]) {
      pluginConfig = context.plugins[PLUGIN_NAME];
    } else if (context && context.pluginConfig) {
      pluginConfig = context.pluginConfig;
    }

    // Reset pending requests tracking
    pendingRequests = {};

    return null;
  }

  /* ------------------------------------------------------------------ */
  /*  Register with the QuikForms plugin framework                      */
  /* ------------------------------------------------------------------ */

  if (typeof QuikFormsPlugins !== 'undefined' && QuikFormsPlugins.register) {
    QuikFormsPlugins.register(PLUGIN_NAME, {
      hooks: {
        onFormLoad: onFormLoad,
        onFieldBlur: onFieldBlur
      }
    });
  }

})();

/**
 * Entry point called by the QuikForms plugin loader after the script loads.
 * The IIFE above has already registered the plugin if QuikFormsPlugins was
 * available at parse time. This function acts as a safety net for cases
 * where the global was not yet available (deferred loading).
 *
 * @param {object} registry - The QuikFormsPlugins registry instance
 */
function initQuikFormsExternalVerification(registry) {
  if (registry && registry.register && !registry.plugins['externalVerification']) {
    registry.register('externalVerification', {
      hooks: {
        onFormLoad: function() { return null; },
        onFieldBlur: function() { return null; }
      }
    });
    console.warn('QuikForms ExternalVerification: Registered via entry point fallback. IIFE registration may have failed.');
  }
}
