/**
 * QuikForms Progress Indicator Plugin v1.0.0
 * Shows form completion progress based on required field completion.
 *
 * Display modes:
 *   - bar      : Percentage progress bar with animated fill
 *   - dots     : Step dots (one per required field, filled when completed)
 *   - fraction : Text like "4 of 7 fields complete"
 *
 * Configuration (via plugin config or defaults):
 *   mode                : 'bar' | 'dots' | 'fraction'   (default: 'bar')
 *   sticky              : boolean                         (default: false)
 *   showLabel           : boolean                         (default: true)
 *   celebrateOnComplete : boolean                         (default: false)
 *   position            : 'top' | 'bottom'                (default: 'top')
 *
 * Hooks: onFormLoad, onFieldChange, onFieldBlur
 */
(function() {
  'use strict';

  // ── Constants ──────────────────────────────────────────────────────────────

  var PLUGIN_NAME  = 'progressIndicator';
  var CONTAINER_ID = 'qfp-progress';
  var STYLE_ID     = 'qfp-progress-styles';

  var DEFAULTS = {
    mode: 'bar',
    sticky: false,
    showLabel: true,
    celebrateOnComplete: false,
    position: 'top'
  };

  // ── Shared state ───────────────────────────────────────────────────────────

  var state = {
    config: null,
    formData: null,
    previousPercent: -1
  };

  // ── Helpers ────────────────────────────────────────────────────────────────

  /**
   * Merge user-supplied config onto defaults.
   * @param {object|undefined} userConfig
   * @returns {object}
   */
  function resolveConfig(userConfig) {
    var config = {};
    var key;
    for (key in DEFAULTS) {
      if (DEFAULTS.hasOwnProperty(key)) {
        config[key] = DEFAULTS[key];
      }
    }
    if (userConfig && typeof userConfig === 'object') {
      for (key in userConfig) {
        if (userConfig.hasOwnProperty(key) && DEFAULTS.hasOwnProperty(key)) {
          config[key] = userConfig[key];
        }
      }
    }
    return config;
  }

  /**
   * Determine whether a given field value counts as "filled".
   * @param {*} value
   * @param {object} field - The field definition from formData.displayFields
   * @returns {boolean}
   */
  function isFieldFilled(value, field) {
    if (value === null || value === undefined) {
      return false;
    }

    // Checkbox fields: only 'true' or boolean true counts as filled.
    if (field.type === 'inputCheckbox') {
      return value === true || value === 'true';
    }

    // For everything else, a non-empty trimmed string is filled.
    if (typeof value === 'string') {
      return value.trim().length > 0;
    }

    return !!value;
  }

  /**
   * Collect the list of required, visible fields from formData.displayFields.
   * A field is included when:
   *   - field.isRequired === true
   *   - The field row element does NOT have the .hidden class (dependency-hidden)
   *   - The field is a data-entry field (not customHTML)
   *
   * @param {Array} displayFields
   * @returns {Array} Array of field objects that are required and visible
   */
  function getRequiredVisibleFields(displayFields) {
    if (!displayFields || !displayFields.length) {
      return [];
    }

    var result = [];
    for (var i = 0; i < displayFields.length; i++) {
      var field = displayFields[i];

      // Only required data-entry fields
      if (!field.isRequired) continue;
      if (field.type === 'customHTML') continue;

      // Check DOM visibility (dependency-hidden fields get .hidden class)
      var rowEl = document.getElementById('field-row-' + field.id);
      if (rowEl && rowEl.classList.contains('hidden')) continue;

      result.push(field);
    }
    return result;
  }

  /**
   * Read the current value of a field from the DOM, mirroring the approach
   * used by QuikForms core (qfjs.js) getFieldValue.
   * @param {object} field
   * @returns {string|null}
   */
  function getFieldDOMValue(field) {
    var el = document.getElementById('field-' + field.id);
    if (!el) return null;

    if (el.type === 'checkbox') {
      return el.checked ? 'true' : 'false';
    }

    // Custom select / radio / checkbox-group rendered as <div>
    if (el.tagName === 'DIV') {
      if (el.classList.contains('qf-custom-select')) {
        var hiddenInput = el.querySelector('.qf-custom-select-value');
        return hiddenInput ? hiddenInput.value : '';
      }
      if (field.customStructureType === 'selectRadio') {
        var checked = document.querySelector('input[name="radio-' + field.id + '"]:checked');
        return checked ? checked.value : '';
      }
      if (field.customStructureType === 'selectCheckboxes') {
        var boxes = document.querySelectorAll('input[name="checkbox-' + field.id + '"]:checked');
        return boxes.length > 0
          ? Array.prototype.map.call(boxes, function(c) { return c.value; }).join(', ')
          : '';
      }
    }

    return el.value || '';
  }

  /**
   * Calculate progress: how many required visible fields are filled.
   * @param {Array} requiredFields - from getRequiredVisibleFields()
   * @returns {{ filled: number, total: number, percent: number }}
   */
  function calculateProgress(requiredFields) {
    var filled = 0;
    var total  = requiredFields.length;

    for (var i = 0; i < requiredFields.length; i++) {
      var field = requiredFields[i];
      var value = getFieldDOMValue(field);
      if (isFieldFilled(value, field)) {
        filled++;
      }
    }

    var percent = total > 0 ? Math.round((filled / total) * 100) : 100;
    return { filled: filled, total: total, percent: percent };
  }

  // ── DOM Rendering ──────────────────────────────────────────────────────────

  /**
   * Minimal HTML entity escaping for safe text insertion.
   * @param {string} str
   * @returns {string}
   */
  function escapeHTML(str) {
    if (!str) return '';
    return str
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;');
  }

  /**
   * Inject the <style> tag for the progress indicator (only once).
   * Uses CSS custom properties already defined by QuikForms core.
   */
  function injectStyles() {
    if (document.getElementById(STYLE_ID)) return;

    var css = [
      '/* QuikForms Progress Indicator Plugin */',

      '.qfp-progress-container {',
      '  box-sizing: border-box;',
      '  width: 100%;',
      '  padding: 12px 16px;',
      '  font-family: inherit;',
      '  transition: opacity 0.3s ease;',
      '}',

      '.qfp-progress-container.qfp-sticky {',
      '  position: sticky;',
      '  top: 0;',
      '  z-index: 100;',
      '  background: var(--qf-input-bg, #ffffff);',
      '  border-bottom: 1px solid rgba(0, 0, 0, 0.08);',
      '}',

      '.qfp-progress-label {',
      '  font-size: 13px;',
      '  color: var(--qf-text-color, #333333);',
      '  margin-bottom: 6px;',
      '  font-weight: 500;',
      '  line-height: 1.3;',
      '}',
      '.qfp-progress-label.qfp-hidden {',
      '  display: none;',
      '}',

      '.qfp-progress-track {',
      '  width: 100%;',
      '  height: 8px;',
      '  background: var(--qf-input-bg, #e0e0e0);',
      '  border-radius: var(--qf-radius-sm, 4px);',
      '  overflow: hidden;',
      '}',

      '.qfp-progress-fill {',
      '  height: 100%;',
      '  width: 0%;',
      '  background: var(--qf-highlight-color, #0070d2);',
      '  border-radius: var(--qf-radius-sm, 4px);',
      '  transition: width 0.4s ease;',
      '}',

      '.qfp-progress-dots {',
      '  display: flex;',
      '  flex-wrap: wrap;',
      '  align-items: center;',
      '  gap: 8px;',
      '}',
      '.qfp-progress-dots .qfp-progress-label {',
      '  width: 100%;',
      '  margin-bottom: 2px;',
      '}',

      '.qfp-dot {',
      '  width: 12px;',
      '  height: 12px;',
      '  border-radius: 50%;',
      '  background: var(--qf-input-bg, #e0e0e0);',
      '  border: 2px solid var(--qf-highlight-color, #0070d2);',
      '  transition: background 0.3s ease, transform 0.3s ease;',
      '}',
      '.qfp-dot-filled {',
      '  background: var(--qf-highlight-color, #0070d2);',
      '  transform: scale(1.1);',
      '}',

      '.qfp-progress-fraction {',
      '  font-size: 15px;',
      '  font-weight: 600;',
      '  color: var(--qf-text-color, #333333);',
      '  line-height: 1.4;',
      '}',
      '.qfp-progress-fraction .qfp-fraction-highlight {',
      '  color: var(--qf-highlight-color, #0070d2);',
      '}',

      '@keyframes qfp-celebrate {',
      '  0%   { transform: scale(1); }',
      '  50%  { transform: scale(1.05); }',
      '  100% { transform: scale(1); }',
      '}',
      '.qfp-celebrate {',
      '  animation: qfp-celebrate 0.5s ease;',
      '}'
    ].join('\n');

    var styleEl = document.createElement('style');
    styleEl.id = STYLE_ID;
    styleEl.type = 'text/css';
    styleEl.textContent = css;
    document.head.appendChild(styleEl);
  }

  /**
   * Build inner HTML for bar mode.
   */
  function buildBarHTML(progress, labelHTML) {
    var html = labelHTML;
    html += '<div class="qfp-progress-track">';
    html += '<div class="qfp-progress-fill" style="width: ' + progress.percent + '%"></div>';
    html += '</div>';
    return html;
  }

  /**
   * Build inner HTML for dots mode.
   */
  function buildDotsHTML(progress, labelHTML, requiredFields) {
    var html = labelHTML;

    for (var i = 0; i < requiredFields.length; i++) {
      var field = requiredFields[i];
      var value = getFieldDOMValue(field);
      var filled = isFieldFilled(value, field);
      html += '<div class="qfp-dot' + (filled ? ' qfp-dot-filled' : '') + '"';
      html += ' title="' + escapeHTML(field.label || '') + '"';
      html += '></div>';
    }

    return html;
  }

  /**
   * Build inner HTML for fraction mode.
   */
  function buildFractionHTML(progress, labelHTML) {
    var html = labelHTML;
    html += '<div class="qfp-progress-fraction">';
    html += '<span class="qfp-fraction-highlight">' + progress.filled + '</span>';
    html += ' of ' + progress.total + ' fields complete';
    html += '</div>';
    return html;
  }

  /**
   * Build the inner HTML for the progress indicator depending on mode.
   * @param {object} progress  - { filled, total, percent }
   * @param {object} config
   * @param {Array}  requiredFields
   * @returns {string}
   */
  function buildProgressHTML(progress, config, requiredFields) {
    var labelText  = progress.filled + ' of ' + progress.total + ' fields complete';
    var labelClass = 'qfp-progress-label' + (config.showLabel ? '' : ' qfp-hidden');
    var labelHTML  = '<div class="' + labelClass + '">' + escapeHTML(labelText) + '</div>';

    switch (config.mode) {
      case 'dots':
        return buildDotsHTML(progress, labelHTML, requiredFields);
      case 'fraction':
        return buildFractionHTML(progress, labelHTML);
      case 'bar':
      default:
        return buildBarHTML(progress, labelHTML);
    }
  }

  /**
   * Create (or find) the container element and inject/update progress UI.
   * @param {object} progress
   * @param {object} config
   * @param {Array}  requiredFields
   */
  function renderProgressIndicator(progress, config, requiredFields) {
    var container = document.getElementById(CONTAINER_ID);

    if (!container) {
      container = document.createElement('div');
      container.id = CONTAINER_ID;
      container.className = 'qfp-progress-container';

      if (config.mode === 'dots') {
        container.className += ' qfp-progress-dots';
      }
      if (config.sticky) {
        container.className += ' qfp-sticky';
      }

      // Find the form container to prepend/append to
      var formContainer = document.querySelector('.qf-fields-container')
                       || document.getElementById('qf-fields-container')
                       || document.querySelector('.qf-form-container')
                       || document.getElementById('qf-form-container');

      if (formContainer) {
        if (config.position === 'bottom') {
          formContainer.appendChild(container);
        } else {
          formContainer.insertBefore(container, formContainer.firstChild);
        }
      } else {
        // Fallback: append to body
        document.body.appendChild(container);
      }
    }

    container.innerHTML = buildProgressHTML(progress, config, requiredFields);
  }

  /**
   * Trigger a brief celebration animation when progress reaches 100%.
   * @param {object} config
   */
  function triggerCelebration(config) {
    if (!config.celebrateOnComplete) return;

    var container = document.getElementById(CONTAINER_ID);
    if (!container) return;

    // Remove then re-add to restart the animation
    container.classList.remove('qfp-celebrate');
    void container.offsetWidth; // force reflow
    container.classList.add('qfp-celebrate');
  }

  // ── Hook Handlers ──────────────────────────────────────────────────────────

  /**
   * onFormLoad hook handler.
   * Reads configuration, injects styles, counts required visible fields,
   * calculates initial progress, and renders the progress indicator.
   *
   * @param {object} context - { formData, fieldValues, locale }
   */
  function handleFormLoad(context) {
    if (!context || !context.formData) return;

    state.formData = context.formData;

    // Resolve configuration from plugin config on formData (if provided)
    var pluginConfig = null;
    if (context.formData.pluginConfig && context.formData.pluginConfig[PLUGIN_NAME]) {
      pluginConfig = context.formData.pluginConfig[PLUGIN_NAME];
    }
    state.config = resolveConfig(pluginConfig);

    // Inject styles
    injectStyles();

    // Calculate initial progress (some fields may have default values)
    var requiredFields = getRequiredVisibleFields(state.formData.displayFields);
    var progress = calculateProgress(requiredFields);
    state.previousPercent = progress.percent;

    // Render
    renderProgressIndicator(progress, state.config, requiredFields);
  }

  /**
   * Shared update logic for onFieldChange and onFieldBlur.
   * Recalculates progress from current DOM values and updates the indicator.
   *
   * @param {object} context - { fieldId, field, value, formData, locale }
   */
  function handleUpdate(context) {
    if (!state.config || !state.formData) return;

    // Accept the latest formData (dependencies may have toggled field visibility)
    if (context && context.formData) {
      state.formData = context.formData;
    }

    var requiredFields = getRequiredVisibleFields(state.formData.displayFields);
    var progress = calculateProgress(requiredFields);

    renderProgressIndicator(progress, state.config, requiredFields);

    // Celebrate on first transition to 100%
    if (progress.percent === 100 && state.previousPercent < 100) {
      triggerCelebration(state.config);
    }

    state.previousPercent = progress.percent;
  }

  // ── Build plugin definition object ─────────────────────────────────────────

  var pluginDef = {
    fieldTypes: {},
    hooks: {
      onFormLoad:    handleFormLoad,
      onFieldChange: handleUpdate,
      onFieldBlur:   handleUpdate
    }
  };

  // ── Auto-registration ──────────────────────────────────────────────────────
  // Register immediately if QuikFormsPlugins is already on the global scope
  // (standard case when qfjs.js loads before plugin resources).

  if (typeof QuikFormsPlugins !== 'undefined' && QuikFormsPlugins.register) {
    QuikFormsPlugins.register(PLUGIN_NAME, pluginDef);
  }

  // ── Expose for entry-point registration ────────────────────────────────────
  // The plugin loader calls window[entryPoint](registry) after the script
  // loads. We store the definition on a namespaced global so the entry point
  // function (declared outside the IIFE) can access it without duplicating
  // the implementation.

  window.__quikFormsProgressIndicatorDef = pluginDef;

})();

/**
 * Entry point called by the QuikForms plugin loader after the script is loaded.
 * Signature: window[entryPoint](registry)
 *
 * If the IIFE above already self-registered (QuikFormsPlugins was global at
 * parse time), this is a no-op. Otherwise it performs registration via the
 * supplied registry reference, reusing the same plugin definition object.
 *
 * @param {object} registry - The QuikFormsPlugins registry instance
 */
function initQuikFormsProgressIndicator(registry) {
  if (!registry || typeof registry.register !== 'function') {
    return;
  }

  // Skip if the IIFE already registered successfully
  if (registry.plugins && registry.plugins['progressIndicator']) {
    return;
  }

  // Use the definition object created by the IIFE
  var def = window.__quikFormsProgressIndicatorDef;
  if (def) {
    registry.register('progressIndicator', def);
  }
}
