/**
 * QuikForms Range Slider Plugin
 * A custom field type plugin that renders an interactive range slider
 * with value bubble, optional tick marks, and full theme integration.
 *
 * @version 1.0.0
 * @license MIT
 */
(function() {
  'use strict';

  /* ------------------------------------------------------------------ */
  /*  Helpers                                                           */
  /* ------------------------------------------------------------------ */

  /**
   * Parse the JSON configuration stored in field.description.
   * Falls back to sensible defaults when the value is missing or invalid.
   *
   * @param {object} field - The field definition provided by QuikForms
   * @returns {object} Parsed configuration with all keys guaranteed
   */
  function parseConfig(field) {
    var defaults = {
      min: 0,
      max: 100,
      step: 1,
      unit: '',
      defaultValue: 50,
      showTicks: false,
      tickInterval: 10
    };

    if (!field || !field.description) {
      return defaults;
    }

    try {
      var parsed = JSON.parse(field.description);
      return {
        min:          parsed.min          !== undefined ? Number(parsed.min)          : defaults.min,
        max:          parsed.max          !== undefined ? Number(parsed.max)          : defaults.max,
        step:         parsed.step         !== undefined ? Number(parsed.step)         : defaults.step,
        unit:         parsed.unit         !== undefined ? String(parsed.unit)         : defaults.unit,
        defaultValue: parsed.defaultValue !== undefined ? Number(parsed.defaultValue) : defaults.defaultValue,
        showTicks:    parsed.showTicks    !== undefined ? Boolean(parsed.showTicks)   : defaults.showTicks,
        tickInterval: parsed.tickInterval !== undefined ? Number(parsed.tickInterval) : defaults.tickInterval
      };
    } catch (e) {
      console.warn('QuikForms RangeSlider: Could not parse field config, using defaults.', e);
      return defaults;
    }
  }

  /**
   * Calculate the fill percentage for a given value within a range.
   *
   * @param {number} value
   * @param {number} min
   * @param {number} max
   * @returns {number} Percentage (0-100)
   */
  function calcPercent(value, min, max) {
    if (max === min) return 0;
    return ((value - min) / (max - min)) * 100;
  }

  /**
   * Build the tick marks HTML when showTicks is enabled.
   *
   * @param {object} cfg - Parsed config
   * @returns {string} HTML string of tick marks
   */
  function buildTicks(cfg) {
    if (!cfg.showTicks || cfg.tickInterval <= 0) return '';

    var ticks = [];
    for (var v = cfg.min; v <= cfg.max; v += cfg.tickInterval) {
      var pct = calcPercent(v, cfg.min, cfg.max);
      ticks.push(
        '<div class="qfp-slider-tick" style="left:' + pct + '%">' +
          '<span class="qfp-slider-tick-mark"></span>' +
          '<span class="qfp-slider-tick-label">' + v + '</span>' +
        '</div>'
      );
    }
    return '<div class="qfp-slider-ticks">' + ticks.join('') + '</div>';
  }

  /**
   * Build a unique scoped <style> block for the slider.
   * Uses CSS custom properties from the QuikForms theme so the slider
   * adapts automatically to any configured colour scheme.
   *
   * @param {string} fieldId - Unique field identifier used for scoping
   * @returns {string} HTML <style> element
   */
  function buildStyles(fieldId) {
    var scope = '#qfp-slider-' + fieldId;

    return '<style data-qfp-slider="' + fieldId + '">' +

      /* Container */
      scope + '.qfp-slider-container {' +
        'position: relative;' +
        'padding: 28px 0 8px;' +
        'user-select: none;' +
        '-webkit-user-select: none;' +
      '}' +

      /* Track wrapper — provides the visible track behind the native input */
      scope + ' .qfp-slider-track-wrapper {' +
        'position: relative;' +
        'height: 8px;' +
        'border-radius: var(--qf-radius-sm, 12px);' +
        'background: var(--qf-input-bg, rgba(30, 41, 59, 0.6));' +
        'border: 1px solid var(--qf-input-border-color, var(--qf-border, rgba(148,163,184,0.2)));' +
        'overflow: visible;' +
      '}' +

      /* Filled portion of the track */
      scope + ' .qfp-slider-fill {' +
        'position: absolute;' +
        'top: 0;' +
        'left: 0;' +
        'height: 100%;' +
        'border-radius: var(--qf-radius-sm, 12px);' +
        'background: var(--qf-highlight-color, #7c3aed);' +
        'transition: width 0.05s ease;' +
        'pointer-events: none;' +
      '}' +

      /* Native range input — stretched over the track wrapper */
      scope + ' input[type="range"] {' +
        'position: absolute;' +
        'top: 50%;' +
        'left: 0;' +
        'width: 100%;' +
        'height: 100%;' +
        'margin: 0;' +
        'transform: translateY(-50%);' +
        '-webkit-appearance: none;' +
        'appearance: none;' +
        'background: transparent;' +
        'cursor: pointer;' +
        'z-index: 2;' +
        'outline: none;' +
      '}' +

      /* Webkit track — hidden (our custom track wrapper is visible instead) */
      scope + ' input[type="range"]::-webkit-slider-runnable-track {' +
        'height: 8px;' +
        'background: transparent;' +
        'border: none;' +
      '}' +

      /* Webkit thumb */
      scope + ' input[type="range"]::-webkit-slider-thumb {' +
        '-webkit-appearance: none;' +
        'appearance: none;' +
        'width: 22px;' +
        'height: 22px;' +
        'border-radius: 50%;' +
        'background: var(--qf-highlight-color, #7c3aed);' +
        'border: 3px solid #fff;' +
        'box-shadow: 0 2px 6px rgba(0,0,0,0.25);' +
        'margin-top: -7px;' +
        'transition: transform 0.15s ease, box-shadow 0.15s ease;' +
      '}' +

      scope + ' input[type="range"]::-webkit-slider-thumb:hover {' +
        'transform: scale(1.15);' +
        'box-shadow: 0 3px 10px rgba(0,0,0,0.3);' +
      '}' +

      scope + ' input[type="range"]:active::-webkit-slider-thumb {' +
        'transform: scale(1.25);' +
      '}' +

      /* Firefox track */
      scope + ' input[type="range"]::-moz-range-track {' +
        'height: 8px;' +
        'background: transparent;' +
        'border: none;' +
      '}' +

      /* Firefox thumb */
      scope + ' input[type="range"]::-moz-range-thumb {' +
        'width: 18px;' +
        'height: 18px;' +
        'border-radius: 50%;' +
        'background: var(--qf-highlight-color, #7c3aed);' +
        'border: 3px solid #fff;' +
        'box-shadow: 0 2px 6px rgba(0,0,0,0.25);' +
        'transition: transform 0.15s ease, box-shadow 0.15s ease;' +
      '}' +

      scope + ' input[type="range"]::-moz-range-thumb:hover {' +
        'transform: scale(1.15);' +
        'box-shadow: 0 3px 10px rgba(0,0,0,0.3);' +
      '}' +

      /* Focus ring */
      scope + ' input[type="range"]:focus-visible::-webkit-slider-thumb {' +
        'outline: 2px solid var(--qf-highlight-color, #7c3aed);' +
        'outline-offset: 3px;' +
      '}' +

      scope + ' input[type="range"]:focus-visible::-moz-range-thumb {' +
        'outline: 2px solid var(--qf-highlight-color, #7c3aed);' +
        'outline-offset: 3px;' +
      '}' +

      /* Value bubble */
      scope + ' .qfp-slider-bubble {' +
        'position: absolute;' +
        'top: -2px;' +
        'transform: translateX(-50%);' +
        'background: var(--qf-highlight-color, #7c3aed);' +
        'color: #fff;' +
        'font-size: 12px;' +
        'font-weight: 600;' +
        'padding: 3px 8px;' +
        'border-radius: var(--qf-radius-sm, 12px);' +
        'white-space: nowrap;' +
        'pointer-events: none;' +
        'opacity: 0;' +
        'transition: opacity 0.2s ease, transform 0.15s ease;' +
        'z-index: 3;' +
      '}' +

      scope + ' .qfp-slider-bubble::after {' +
        'content: "";' +
        'position: absolute;' +
        'bottom: -4px;' +
        'left: 50%;' +
        'transform: translateX(-50%);' +
        'width: 0;' +
        'height: 0;' +
        'border-left: 5px solid transparent;' +
        'border-right: 5px solid transparent;' +
        'border-top: 5px solid var(--qf-highlight-color, #7c3aed);' +
      '}' +

      /* Show bubble on hover / active / focus */
      scope + ':hover .qfp-slider-bubble,' +
      scope + '.qfp-slider--active .qfp-slider-bubble {' +
        'opacity: 1;' +
        'transform: translateX(-50%) translateY(-4px);' +
      '}' +

      /* Current value display (always visible below the slider) */
      scope + ' .qfp-slider-value {' +
        'text-align: center;' +
        'margin-top: 8px;' +
        'font-size: 14px;' +
        'font-weight: 500;' +
        'color: var(--qf-input-text, #f1f5f9);' +
        'letter-spacing: 0.02em;' +
      '}' +

      /* Min / max labels */
      scope + ' .qfp-slider-limits {' +
        'display: flex;' +
        'justify-content: space-between;' +
        'margin-top: 4px;' +
        'font-size: 11px;' +
        'color: var(--qf-desc-text, #94a3b8);' +
      '}' +

      /* Tick marks */
      scope + ' .qfp-slider-ticks {' +
        'position: relative;' +
        'height: 20px;' +
        'margin-top: 4px;' +
      '}' +

      scope + ' .qfp-slider-tick {' +
        'position: absolute;' +
        'transform: translateX(-50%);' +
        'display: flex;' +
        'flex-direction: column;' +
        'align-items: center;' +
      '}' +

      scope + ' .qfp-slider-tick-mark {' +
        'display: block;' +
        'width: 1px;' +
        'height: 6px;' +
        'background: var(--qf-input-border-color, var(--qf-border, rgba(148,163,184,0.2)));' +
      '}' +

      scope + ' .qfp-slider-tick-label {' +
        'font-size: 10px;' +
        'color: var(--qf-desc-text, #94a3b8);' +
        'margin-top: 2px;' +
      '}' +

    '</style>';
  }

  /**
   * Escape a string for safe use inside an HTML attribute value.
   *
   * @param {string} str
   * @returns {string}
   */
  function escAttr(str) {
    return String(str)
      .replace(/&/g, '&amp;')
      .replace(/"/g, '&quot;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;');
  }

  /* ------------------------------------------------------------------ */
  /*  Internal state store                                              */
  /* ------------------------------------------------------------------ */

  /**
   * Stores per-field configuration so initialize / getValue / setValue /
   * validate can reference the original config without re-parsing.
   */
  var fieldConfigs = {};

  /* ------------------------------------------------------------------ */
  /*  Custom field type definition                                      */
  /* ------------------------------------------------------------------ */

  var rangeSliderFieldType = {

    /**
     * Render the range slider HTML for a given field.
     *
     * @param {object} field   - Field definition (id, label, description, required, ...)
     * @param {object} context - Render context supplied by the framework
     * @returns {string} HTML string
     */
    render: function(field, context) {
      var cfg = parseConfig(field);
      fieldConfigs[field.id] = cfg;

      var initialValue = cfg.defaultValue;
      var pct = calcPercent(initialValue, cfg.min, cfg.max);
      var displayValue = initialValue + (cfg.unit ? ' ' + cfg.unit : '');

      var html =
        buildStyles(field.id) +

        '<div id="qfp-slider-' + escAttr(field.id) + '" class="qfp-slider-container">' +

          /* Value bubble (floats above the thumb) */
          '<div class="qfp-slider-bubble" style="left:' + pct + '%">' +
            escAttr(displayValue) +
          '</div>' +

          /* Track wrapper with fill bar and native input */
          '<div class="qfp-slider-track-wrapper">' +
            '<div class="qfp-slider-fill" style="width:' + pct + '%"></div>' +
            '<input type="range"' +
              ' id="qfp-range-' + escAttr(field.id) + '"' +
              ' min="' + escAttr(String(cfg.min)) + '"' +
              ' max="' + escAttr(String(cfg.max)) + '"' +
              ' step="' + escAttr(String(cfg.step)) + '"' +
              ' value="' + escAttr(String(initialValue)) + '"' +
              ' aria-label="' + escAttr(field.label || 'Range slider') + '"' +
              ' aria-valuemin="' + escAttr(String(cfg.min)) + '"' +
              ' aria-valuemax="' + escAttr(String(cfg.max)) + '"' +
              ' aria-valuenow="' + escAttr(String(initialValue)) + '"' +
              (cfg.unit ? ' aria-valuetext="' + escAttr(displayValue) + '"' : '') +
            '>' +
          '</div>' +

          /* Tick marks (optional) */
          buildTicks(cfg) +

          /* Min / max labels */
          '<div class="qfp-slider-limits">' +
            '<span>' + escAttr(String(cfg.min)) + (cfg.unit ? ' ' + escAttr(cfg.unit) : '') + '</span>' +
            '<span>' + escAttr(String(cfg.max)) + (cfg.unit ? ' ' + escAttr(cfg.unit) : '') + '</span>' +
          '</div>' +

          /* Current value text */
          '<div class="qfp-slider-value">' + escAttr(displayValue) + '</div>' +

        '</div>';

      return html;
    },

    /**
     * Initialize interactive behaviour after the field HTML is in the DOM.
     *
     * @param {string}      fieldId   - Unique field identifier
     * @param {HTMLElement}  container - The parent element that contains the rendered HTML
     */
    initialize: function(fieldId, container) {
      var root = container.querySelector('#qfp-slider-' + fieldId);
      if (!root) return;

      var input  = root.querySelector('input[type="range"]');
      var fill   = root.querySelector('.qfp-slider-fill');
      var bubble = root.querySelector('.qfp-slider-bubble');
      var label  = root.querySelector('.qfp-slider-value');

      if (!input) return;

      var cfg = fieldConfigs[fieldId] || parseConfig(null);

      /**
       * Synchronise visual elements with the current input value.
       */
      function syncVisuals() {
        var val = Number(input.value);
        var pct = calcPercent(val, cfg.min, cfg.max);
        var display = val + (cfg.unit ? ' ' + cfg.unit : '');

        if (fill)   fill.style.width = pct + '%';
        if (bubble) {
          bubble.style.left = pct + '%';
          bubble.textContent = display;
        }
        if (label)  label.textContent = display;

        /* ARIA */
        input.setAttribute('aria-valuenow', val);
        if (cfg.unit) {
          input.setAttribute('aria-valuetext', display);
        }
      }

      /* --- Event listeners ------------------------------------------ */

      /* Continuous updates while dragging */
      input.addEventListener('input', function() {
        syncVisuals();
      });

      /* Final value on release — can be used for analytics / hooks */
      input.addEventListener('change', function() {
        syncVisuals();
      });

      /* Show bubble on pointer down / touch start */
      function activate() { root.classList.add('qfp-slider--active'); }
      function deactivate() { root.classList.remove('qfp-slider--active'); }

      input.addEventListener('mousedown',  activate);
      input.addEventListener('touchstart', activate, { passive: true });
      input.addEventListener('mouseup',    deactivate);
      input.addEventListener('touchend',   deactivate);
      input.addEventListener('blur',       deactivate);

      /* Keyboard support — the native range input already handles arrow
         keys, but we need to make sure visuals stay in sync. */
      input.addEventListener('keydown', function() {
        activate();
      });
      input.addEventListener('keyup', function() {
        syncVisuals();
        deactivate();
      });

      /* Initial visual sync (in case the DOM value differs from render) */
      syncVisuals();
    },

    /**
     * Get the current value of the slider.
     *
     * @param {string} fieldId
     * @returns {string} The current value as a string
     */
    getValue: function(fieldId) {
      var input = document.querySelector('#qfp-range-' + fieldId);
      return input ? input.value : '';
    },

    /**
     * Programmatically set the slider value and update the visual state.
     *
     * @param {string} fieldId
     * @param {*}      value - New value (will be cast to Number)
     */
    setValue: function(fieldId, value) {
      var input = document.querySelector('#qfp-range-' + fieldId);
      if (!input) return;

      var cfg = fieldConfigs[fieldId] || parseConfig(null);
      var numVal = Number(value);

      /* Clamp to valid range */
      if (isNaN(numVal)) numVal = cfg.defaultValue;
      numVal = Math.max(cfg.min, Math.min(cfg.max, numVal));

      input.value = numVal;

      /* Trigger visual sync by dispatching an input event */
      var evt;
      try {
        evt = new Event('input', { bubbles: true });
      } catch (e) {
        /* IE11 fallback */
        evt = document.createEvent('Event');
        evt.initEvent('input', true, true);
      }
      input.dispatchEvent(evt);
    },

    /**
     * Validate the current slider value.
     *
     * @param {string} fieldId
     * @param {object} field - The field definition (includes required flag)
     * @returns {object} { isValid: boolean, errors: string[] }
     */
    validate: function(fieldId, field) {
      var errors = [];
      var input = document.querySelector('#qfp-range-' + fieldId);
      var cfg   = fieldConfigs[fieldId] || parseConfig(field);

      if (!input) {
        errors.push('Range slider field not found.');
        return { isValid: false, errors: errors };
      }

      var val = Number(input.value);

      /* Required check — value must differ from the default */
      if (field && field.required) {
        if (isNaN(val)) {
          errors.push((field.label || 'This field') + ' is required.');
        }
      }

      /* Range bounds check */
      if (!isNaN(val)) {
        if (val < cfg.min) {
          errors.push((field.label || 'Value') + ' must be at least ' + cfg.min + '.');
        }
        if (val > cfg.max) {
          errors.push((field.label || 'Value') + ' must be no more than ' + cfg.max + '.');
        }
      }

      return {
        isValid: errors.length === 0,
        errors: errors
      };
    }
  };

  /* ------------------------------------------------------------------ */
  /*  Register with the QuikForms plugin framework                      */
  /* ------------------------------------------------------------------ */

  if (typeof QuikFormsPlugins !== 'undefined' && QuikFormsPlugins.register) {
    QuikFormsPlugins.register('rangeSlider', {
      fieldTypes: {
        rangeSlider: rangeSliderFieldType
      }
    });
  }

})();

/**
 * Entry point called by the QuikForms plugin loader after the script loads.
 * The IIFE above has already registered the plugin if QuikFormsPlugins was
 * available at parse time. This function acts as a safety net for cases
 * where the global was not yet available (deferred loading).
 *
 * @param {object} registry - The QuikFormsPlugins registry instance
 */
function initQuikFormsRangeSlider(registry) {
  if (registry && registry.register && !registry.plugins['rangeSlider']) {
    registry.register('rangeSlider', {
      fieldTypes: {
        rangeSlider: {
          render:     function() { return '<div class="qf-plugin-field-error">Range Slider: plugin failed to load correctly.</div>'; },
          initialize: function() {},
          getValue:   function() { return ''; },
          setValue:    function() {},
          validate:   function() { return { isValid: true, errors: [] }; }
        }
      }
    });
    console.warn('QuikForms RangeSlider: Registered via entry point fallback. IIFE registration may have failed.');
  }
}
