/**
 * QuikForms Smart Validation Suite Plugin
 *
 * Hybrid plugin providing real-time client-side validation UX and server-side
 * enforcement via IQuikFormsFieldValidator. Features include email typo
 * detection, phone validation, cross-field rules, character counters, and
 * inline error display.
 *
 * @version 1.0.0
 * @author QuikForms
 */
(function() {
  'use strict';

  // ---------------------------------------------------------------------------
  // Constants
  // ---------------------------------------------------------------------------

  var KNOWN_EMAIL_DOMAINS = [
    'gmail.com', 'yahoo.com', 'hotmail.com', 'outlook.com', 'aol.com',
    'icloud.com', 'mail.com', 'protonmail.com', 'live.com', 'msn.com'
  ];

  var EMAIL_REGEX = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  var PHONE_DIGITS_MIN = 10;
  var LEVENSHTEIN_THRESHOLD = 2;

  var CSS_PREFIX = 'qf-sv-';
  var SUGGESTION_CLASS = CSS_PREFIX + 'suggestion';
  var INLINE_ERROR_CLASS = CSS_PREFIX + 'inline-error';
  var INLINE_WARNING_CLASS = CSS_PREFIX + 'inline-warning';
  var CHAR_COUNTER_CLASS = CSS_PREFIX + 'char-counter';
  var CHAR_COUNTER_WARN_CLASS = CSS_PREFIX + 'char-warn';
  var CHAR_COUNTER_OVER_CLASS = CSS_PREFIX + 'char-over';

  // ---------------------------------------------------------------------------
  // Utility: Levenshtein Distance
  // ---------------------------------------------------------------------------

  /**
   * Compute the Levenshtein edit distance between two strings.
   * @param {string} a
   * @param {string} b
   * @returns {number}
   */
  function levenshtein(a, b) {
    if (a === b) return 0;
    if (a.length === 0) return b.length;
    if (b.length === 0) return a.length;

    var matrix = [];
    var i, j;

    for (i = 0; i <= b.length; i++) {
      matrix[i] = [i];
    }
    for (j = 0; j <= a.length; j++) {
      matrix[0][j] = j;
    }

    for (i = 1; i <= b.length; i++) {
      for (j = 1; j <= a.length; j++) {
        if (b.charAt(i - 1) === a.charAt(j - 1)) {
          matrix[i][j] = matrix[i - 1][j - 1];
        } else {
          matrix[i][j] = Math.min(
            matrix[i - 1][j - 1] + 1, // substitution
            matrix[i][j - 1] + 1,       // insertion
            matrix[i - 1][j] + 1        // deletion
          );
        }
      }
    }

    return matrix[b.length][a.length];
  }

  // ---------------------------------------------------------------------------
  // Utility: DOM helpers
  // ---------------------------------------------------------------------------

  /**
   * Find the container element for a given field id. Falls back to the input's
   * parentNode if the standard row wrapper is not found.
   * @param {string} fieldId
   * @returns {Element|null}
   */
  function getFieldRow(fieldId) {
    return document.getElementById('field-row-' + fieldId) || null;
  }

  /**
   * Get the actual input / textarea / select element for a field.
   * @param {string} fieldId
   * @returns {Element|null}
   */
  function getFieldElement(fieldId) {
    return document.getElementById('field-' + fieldId) || null;
  }

  /**
   * Remove an element by its id if it exists.
   * @param {string} id
   */
  function removeById(id) {
    var el = document.getElementById(id);
    if (el && el.parentNode) {
      el.parentNode.removeChild(el);
    }
  }

  /**
   * Inject a message div below a field inside its row container.
   * Replaces any existing message with the same id.
   * @param {string} fieldId
   * @param {string} suffixId  - unique suffix to construct the element id
   * @param {string} message   - HTML content
   * @param {string} className - CSS class for the wrapper
   * @returns {Element|null} the created element, or null if row not found
   */
  function injectMessage(fieldId, suffixId, message, className) {
    var id = CSS_PREFIX + suffixId + '-' + fieldId;
    removeById(id);

    var row = getFieldRow(fieldId);
    if (!row) return null;

    var div = document.createElement('div');
    div.id = id;
    div.className = className;
    div.innerHTML = message;
    row.appendChild(div);
    return div;
  }

  /**
   * Remove a previously injected message.
   * @param {string} fieldId
   * @param {string} suffixId
   */
  function removeMessage(fieldId, suffixId) {
    removeById(CSS_PREFIX + suffixId + '-' + fieldId);
  }

  /**
   * Inject the stylesheet for the plugin if it has not already been added.
   */
  function injectStyles() {
    if (document.getElementById(CSS_PREFIX + 'styles')) return;

    var style = document.createElement('style');
    style.id = CSS_PREFIX + 'styles';
    style.textContent = [
      '.' + SUGGESTION_CLASS + ' {',
      '  font-size: 0.85em; color: #1a73e8; margin-top: 4px; cursor: pointer;',
      '}',
      '.' + SUGGESTION_CLASS + ':hover {',
      '  text-decoration: underline;',
      '}',
      '.' + INLINE_ERROR_CLASS + ' {',
      '  font-size: 0.85em; color: #d93025; margin-top: 4px;',
      '}',
      '.' + INLINE_WARNING_CLASS + ' {',
      '  font-size: 0.85em; color: #e37400; margin-top: 4px;',
      '}',
      '.' + CHAR_COUNTER_CLASS + ' {',
      '  font-size: 0.8em; color: #5f6368; margin-top: 2px; text-align: right;',
      '}',
      '.' + CHAR_COUNTER_WARN_CLASS + ' {',
      '  color: #e37400;',
      '}',
      '.' + CHAR_COUNTER_OVER_CLASS + ' {',
      '  color: #d93025; font-weight: 600;',
      '}'
    ].join('\n');
    document.head.appendChild(style);
  }

  // ---------------------------------------------------------------------------
  // Email typo detection
  // ---------------------------------------------------------------------------

  /**
   * Determine whether a field is an email field based on its configuration.
   * @param {object} field - form field descriptor
   * @returns {boolean}
   */
  function isEmailField(field) {
    if (!field) return false;
    var objectField = (field.objectField || '').toLowerCase();
    var fieldType = (field.type || '').toLowerCase();
    var label = (field.label || '').toLowerCase();

    return objectField.indexOf('email') !== -1 ||
           fieldType === 'inputemail' ||
           fieldType === 'email' ||
           label.indexOf('email') !== -1;
  }

  /**
   * Check an email value and return a suggestion if the domain is close to a
   * known domain.
   * @param {string} value
   * @returns {string|null} suggested corrected email, or null
   */
  function getEmailSuggestion(value) {
    if (!value || typeof value !== 'string') return null;

    var parts = value.split('@');
    if (parts.length !== 2) return null;

    var user = parts[0];
    var domain = parts[1].toLowerCase();

    // If domain exactly matches a known domain, no suggestion needed
    if (KNOWN_EMAIL_DOMAINS.indexOf(domain) !== -1) return null;

    var bestMatch = null;
    var bestDistance = LEVENSHTEIN_THRESHOLD + 1;

    for (var i = 0; i < KNOWN_EMAIL_DOMAINS.length; i++) {
      var known = KNOWN_EMAIL_DOMAINS[i];
      var distance = levenshtein(domain, known);
      if (distance > 0 && distance <= LEVENSHTEIN_THRESHOLD && distance < bestDistance) {
        bestDistance = distance;
        bestMatch = known;
      }
    }

    return bestMatch ? (user + '@' + bestMatch) : null;
  }

  /**
   * Run email typo detection on blur and show/hide suggestion.
   * @param {string} fieldId
   * @param {object} field
   * @param {string} value
   */
  function handleEmailBlur(fieldId, field, value) {
    removeMessage(fieldId, 'email-sug');

    if (!isEmailField(field) || !value) return;

    var suggestion = getEmailSuggestion(value);
    if (!suggestion) return;

    var escapedSuggestion = suggestion
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;');

    var html = 'Did you mean <strong>' + escapedSuggestion + '</strong>?';
    var div = injectMessage(fieldId, 'email-sug', html, SUGGESTION_CLASS);

    if (div) {
      div.addEventListener('click', function() {
        var el = getFieldElement(fieldId);
        if (el) {
          el.value = suggestion;
          // Dispatch change event so the form picks up the new value
          var evt;
          try {
            evt = new Event('change', { bubbles: true });
          } catch (_e) {
            evt = document.createEvent('Event');
            evt.initEvent('change', true, true);
          }
          el.dispatchEvent(evt);
        }
        removeMessage(fieldId, 'email-sug');
      });
    }
  }

  // ---------------------------------------------------------------------------
  // Phone validation
  // ---------------------------------------------------------------------------

  /**
   * Determine whether a field is a phone field.
   * @param {object} field
   * @returns {boolean}
   */
  function isPhoneField(field) {
    if (!field) return false;
    var objectField = (field.objectField || '').toLowerCase();
    var fieldType = (field.type || '').toLowerCase();
    var label = (field.label || '').toLowerCase();

    return objectField.indexOf('phone') !== -1 ||
           fieldType === 'inputphone' ||
           fieldType === 'phone' ||
           fieldType === 'tel' ||
           label.indexOf('phone') !== -1;
  }

  /**
   * Count digits in a string.
   * @param {string} value
   * @returns {number}
   */
  function countDigits(value) {
    if (!value) return 0;
    var match = value.match(/\d/g);
    return match ? match.length : 0;
  }

  /**
   * Run phone validation on blur and show/hide inline error.
   * @param {string} fieldId
   * @param {object} field
   * @param {string} value
   */
  function handlePhoneBlur(fieldId, field, value) {
    removeMessage(fieldId, 'phone-err');

    if (!isPhoneField(field) || !value) return;

    var digitCount = countDigits(value);
    if (digitCount > 0 && digitCount < PHONE_DIGITS_MIN) {
      injectMessage(
        fieldId,
        'phone-err',
        'Phone number must contain at least ' + PHONE_DIGITS_MIN + ' digits.',
        INLINE_ERROR_CLASS
      );
    }
  }

  // ---------------------------------------------------------------------------
  // Character counter
  // ---------------------------------------------------------------------------

  /**
   * Initialise character counters for configured fields.
   * @param {object} pluginConfig - plugin configuration from form data
   * @param {Array} displayFields - all form fields
   */
  function initCharCounters(pluginConfig, displayFields) {
    var charLimits = (pluginConfig && pluginConfig.characterLimits) || {};
    // charLimits is a map of fieldId => maxChars (number)

    if (!displayFields) return;

    displayFields.forEach(function(field) {
      var maxChars = charLimits[field.id];
      if (!maxChars) return;

      maxChars = parseInt(maxChars, 10);
      if (isNaN(maxChars) || maxChars <= 0) return;

      // Store on the field for later reference
      field._svMaxChars = maxChars;

      // Render initial counter
      updateCharCounter(field.id, '', maxChars);

      // Attach input listener for real-time counting
      var el = getFieldElement(field.id);
      if (el) {
        el.addEventListener('input', function() {
          updateCharCounter(field.id, el.value || '', maxChars);
        });
      }
    });
  }

  /**
   * Update or create the character counter display for a field.
   * @param {string} fieldId
   * @param {string} value
   * @param {number} maxChars
   */
  function updateCharCounter(fieldId, value, maxChars) {
    var currentLength = (value || '').length;
    var ratio = currentLength / maxChars;

    var id = CSS_PREFIX + 'charcnt-' + fieldId;
    var counter = document.getElementById(id);

    if (!counter) {
      var row = getFieldRow(fieldId);
      if (!row) return;
      counter = document.createElement('div');
      counter.id = id;
      counter.className = CHAR_COUNTER_CLASS;
      row.appendChild(counter);
    }

    counter.textContent = currentLength + ' / ' + maxChars + ' characters';

    // Reset classes
    counter.classList.remove(CHAR_COUNTER_WARN_CLASS, CHAR_COUNTER_OVER_CLASS);

    if (ratio >= 1) {
      counter.classList.add(CHAR_COUNTER_OVER_CLASS);
    } else if (ratio >= 0.8) {
      counter.classList.add(CHAR_COUNTER_WARN_CLASS);
    }
  }

  /**
   * Handle character limit check on blur - show warning if over limit.
   * @param {string} fieldId
   * @param {object} field
   * @param {string} value
   */
  function handleCharLimitBlur(fieldId, field, value) {
    removeMessage(fieldId, 'charlimit-err');

    if (!field._svMaxChars) return;

    var currentLength = (value || '').length;
    if (currentLength > field._svMaxChars) {
      injectMessage(
        fieldId,
        'charlimit-err',
        'Maximum ' + field._svMaxChars + ' characters allowed (' + currentLength + ' entered).',
        INLINE_ERROR_CLASS
      );
    }
  }

  // ---------------------------------------------------------------------------
  // Cross-field rules
  // ---------------------------------------------------------------------------

  /**
   * Run all configured cross-field validation rules.
   * @param {Array} rules - array of rule objects from config
   * @param {object} fieldValues - map of fieldId => value
   * @returns {{ errors: string[], warnings: string[] }}
   */
  function runCrossFieldRules(rules, fieldValues) {
    var errors = [];
    var warnings = [];

    if (!rules || !Array.isArray(rules)) {
      return { errors: errors, warnings: warnings };
    }

    rules.forEach(function(rule) {
      switch (rule.type) {

        case 'match':
          if (rule.fields && rule.fields.length >= 2) {
            var firstVal = fieldValues[rule.fields[0]];
            for (var i = 1; i < rule.fields.length; i++) {
              if (fieldValues[rule.fields[i]] !== firstVal) {
                errors.push(rule.message || 'Fields must match.');
                break;
              }
            }
          }
          break;

        case 'dateAfter':
          if (rule.fields && rule.fields.length >= 2) {
            var startStr = fieldValues[rule.fields[0]];
            var endStr = fieldValues[rule.fields[1]];
            if (startStr && endStr) {
              var startDate = new Date(startStr);
              var endDate = new Date(endStr);
              if (!isNaN(startDate.getTime()) && !isNaN(endDate.getTime()) && endDate <= startDate) {
                errors.push(rule.message || 'End date must be after start date.');
              }
            }
          }
          break;

        case 'requiredIf':
          if (rule.field && rule.condition) {
            var condField = rule.condition.field;
            var condValue = rule.condition.value;
            var condActual = fieldValues[condField];

            // Coerce to string for comparison
            if (String(condActual) === String(condValue)) {
              var targetValue = fieldValues[rule.field];
              if (targetValue === null || targetValue === undefined || String(targetValue).trim() === '') {
                errors.push(rule.message || 'This field is required.');
              }
            }
          }
          break;

        default:
          // Unknown rule type - skip
          break;
      }
    });

    return { errors: errors, warnings: warnings };
  }

  // ---------------------------------------------------------------------------
  // Inline error clearing
  // ---------------------------------------------------------------------------

  /**
   * Clear all inline messages for a given field.
   * @param {string} fieldId
   */
  function clearInlineMessages(fieldId) {
    removeMessage(fieldId, 'email-sug');
    removeMessage(fieldId, 'phone-err');
    removeMessage(fieldId, 'charlimit-err');
    removeMessage(fieldId, 'inline-err');
  }

  // ---------------------------------------------------------------------------
  // Plugin state
  // ---------------------------------------------------------------------------

  var pluginState = {
    config: null,
    displayFields: null,
    fieldMap: null // fieldId => field descriptor
  };

  /**
   * Build a map of fieldId => field descriptor for quick lookup.
   * @param {Array} fields
   * @returns {object}
   */
  function buildFieldMap(fields) {
    var map = {};
    if (fields && Array.isArray(fields)) {
      fields.forEach(function(f) {
        if (f && f.id) {
          map[f.id] = f;
        }
      });
    }
    return map;
  }

  // ---------------------------------------------------------------------------
  // Plugin registration
  // ---------------------------------------------------------------------------

  QuikFormsPlugins.register('smartValidation', {
    hooks: {

      /**
       * onFormLoad: Parse config, identify email/phone fields, set up
       * character counters, inject styles.
       */
      onFormLoad: function(context) {
        injectStyles();

        var formData = context.formData || {};
        var pluginConfig = {};

        // Extract plugin-specific configuration
        if (formData.pluginConfigurations && formData.pluginConfigurations.smartValidation) {
          pluginConfig = formData.pluginConfigurations.smartValidation;
        } else if (formData.pluginConfiguration) {
          pluginConfig = formData.pluginConfiguration;
        }

        pluginState.config = pluginConfig;
        pluginState.displayFields = formData.displayFields || [];
        pluginState.fieldMap = buildFieldMap(pluginState.displayFields);

        // Initialise character counters for configured fields
        initCharCounters(pluginConfig, pluginState.displayFields);
      },

      /**
       * onFieldBlur: Run email typo check, phone validation, and character
       * limit check for the blurred field.
       */
      onFieldBlur: function(context) {
        var fieldId = context.fieldId;
        var field = context.field || (pluginState.fieldMap && pluginState.fieldMap[fieldId]);
        var value = context.value;

        if (!field) return;

        // Email typo detection
        handleEmailBlur(fieldId, field, value);

        // Phone validation
        handlePhoneBlur(fieldId, field, value);

        // Character limit warning
        handleCharLimitBlur(fieldId, field, value);
      },

      /**
       * onFieldChange: Clear inline errors for the changed field.
       */
      onFieldChange: function(context) {
        var fieldId = context.fieldId;
        if (fieldId) {
          clearInlineMessages(fieldId);
        }
      },

      /**
       * onValidate: Run all cross-field rules and per-field validation.
       * Returns aggregated errors and warnings.
       */
      onValidate: function(context) {
        var fieldValues = context.fieldValues || {};
        var errors = [];
        var warnings = [];

        // Per-field validation (email format, phone format, char limits)
        var fields = pluginState.displayFields || [];
        fields.forEach(function(field) {
          var value = fieldValues[field.id];
          if (value === null || value === undefined || String(value).trim() === '') {
            return; // Skip empty - required check is not this plugin's job
          }

          var strValue = String(value);

          // Email format validation
          if (isEmailField(field) && !EMAIL_REGEX.test(strValue)) {
            errors.push((field.label || field.id) + ': Please enter a valid email address.');
          }

          // Phone digit count validation
          if (isPhoneField(field)) {
            var digits = countDigits(strValue);
            if (digits > 0 && digits < PHONE_DIGITS_MIN) {
              errors.push((field.label || field.id) + ': Phone number must contain at least ' + PHONE_DIGITS_MIN + ' digits.');
            }
          }

          // Character limit validation
          if (field._svMaxChars && strValue.length > field._svMaxChars) {
            errors.push(
              (field.label || field.id) + ': Maximum ' + field._svMaxChars +
              ' characters allowed (' + strValue.length + ' entered).'
            );
          }
        });

        // Cross-field rules
        var crossFieldRules = (pluginState.config && pluginState.config.crossFieldRules) || [];
        var crossResult = runCrossFieldRules(crossFieldRules, fieldValues);
        errors = errors.concat(crossResult.errors);
        warnings = warnings.concat(crossResult.warnings);

        return {
          errors: errors,
          warnings: warnings
        };
      }
    }
  });
})();

/**
 * Entry point called by the QuikForms plugin loader after the static resource
 * is loaded. The plugin self-registers via QuikFormsPlugins.register() in the
 * IIFE above, so this function is intentionally minimal.
 *
 * @param {object} registry - the QuikFormsPlugins registry instance
 */
function initQuikFormsSmartValidation(registry) {
  // Plugin is already registered by the IIFE.
  // This entry point exists to satisfy the plugin loader contract.
  if (registry && registry.plugins && registry.plugins.smartValidation) {
    console.log('QuikFormsSmartValidation: Plugin loaded successfully.');
  }
}
