/**
 * @description Submission Enrichment plugin for QuikForms.
 * Automatically enriches form submissions with metadata (timestamp, user agent,
 * IP address, referrer), applies case transformations to field values, and
 * optionally creates follow-up Tasks linked to the submitted record.
 *
 * Configuration keys:
 *   timestampField      - Field API name to stamp submission timestamp (e.g., 'Description')
 *   userAgentField      - Field API name for browser user agent string
 *   ipAddressField      - Field API name for submitter IP address
 *   referrerField       - Field API name for HTTP referrer URL
 *   caseTransformFields - Map of field API names to transform type:
 *                         'uppercase', 'lowercase', or 'titlecase'
 *   createFollowUpTask  - Boolean, whether to create a Task after submission
 *   taskSubject         - Subject line for the follow-up Task
 *   taskDaysUntilDue    - Number of days until Task is due (default: 3)
 *
 * @author QuikForms
 * @since 1.0
 */
global class QuikFormsSubmissionEnrichmentPlugin implements IQuikFormsSubmissionHandler, IQuikFormsPlugin {

    private Map<String, Object> config;
    private Boolean initialized = false;

    // ==================== IQuikFormsPlugin Methods ====================

    /**
     * @description Returns metadata about this plugin
     * @return QuikFormsPluginInfo containing plugin metadata
     */
    global QuikFormsPluginInfo getPluginInfo() {
        QuikFormsPluginInfo info = new QuikFormsPluginInfo();
        info.name = 'Submission Enrichment';
        info.version = '1.0.0';
        info.description = 'Enriches form submissions with metadata stamping, case transformations, and optional follow-up Task creation.';
        info.supportsCallouts = false;
        info.supportedHooks.add('onBeforeSubmit');
        info.supportedHooks.add('onAfterSubmit');
        return info;
    }

    /**
     * @description Initializes the plugin with configuration from the form definition
     * @param configuration Map of configuration options
     */
    global void initialize(Map<String, Object> configuration) {
        this.config = configuration != null ? configuration : new Map<String, Object>();
        this.initialized = true;
    }

    /**
     * @description Returns whether the plugin is properly initialized and ready
     * @return Boolean indicating readiness
     */
    global Boolean isReady() {
        return this.initialized;
    }

    // ==================== IQuikFormsSubmissionHandler Methods ====================

    /**
     * @description Called before record insertion. Stamps metadata fields and applies
     * case transformations to configured field values.
     * @param context Submission context with form data and field values
     * @return QuikFormsSubmissionResult with modified field values and metadata
     */
    global QuikFormsSubmissionResult onBeforeSubmit(QuikFormsSubmissionContext context) {
        QuikFormsSubmissionResult result = new QuikFormsSubmissionResult();

        // 1. Stamp submission metadata into configured fields
        if (config.containsKey('timestampField')) {
            String timestampField = (String) config.get('timestampField');
            if (String.isNotBlank(timestampField)) {
                result.modifyFieldValue(timestampField, Datetime.now().format());
            }
        }

        if (config.containsKey('userAgentField')) {
            String userAgentField = (String) config.get('userAgentField');
            if (String.isNotBlank(userAgentField)) {
                result.modifyFieldValue(userAgentField, context.userAgent);
            }
        }

        if (config.containsKey('ipAddressField')) {
            String ipAddressField = (String) config.get('ipAddressField');
            if (String.isNotBlank(ipAddressField)) {
                result.modifyFieldValue(ipAddressField, context.ipAddress);
            }
        }

        if (config.containsKey('referrerField')) {
            String referrerField = (String) config.get('referrerField');
            if (String.isNotBlank(referrerField)) {
                result.modifyFieldValue(referrerField, context.referrer);
            }
        }

        // 2. Case transformations
        if (config.containsKey('caseTransformFields')) {
            Map<String, Object> transformMap = (Map<String, Object>) config.get('caseTransformFields');
            if (transformMap != null) {
                for (String fieldName : transformMap.keySet()) {
                    String transformType = (String) transformMap.get(fieldName);
                    Object rawValue = context.fieldValues.get(fieldName);

                    if (rawValue != null && rawValue instanceof String) {
                        String fieldValue = (String) rawValue;
                        String transformed = applyTransform(fieldValue, transformType);
                        result.modifyFieldValue(fieldName, transformed);
                    }
                }
            }
        }

        // 3. Pass metadata to onAfterSubmit
        result.setMetadata('enrichmentApplied', true);
        result.setMetadata('enrichmentTimestamp', String.valueOf(Datetime.now()));

        return result;
    }

    /**
     * @description Called after successful record insertion. Creates a follow-up
     * Task linked to the submitted record if configured.
     * @param context Submission context with recordId populated
     * @return QuikFormsSubmissionResult with task creation metadata
     */
    global QuikFormsSubmissionResult onAfterSubmit(QuikFormsSubmissionContext context) {
        QuikFormsSubmissionResult result = new QuikFormsSubmissionResult();

        Boolean createTask = config.containsKey('createFollowUpTask')
            && (Boolean) config.get('createFollowUpTask');

        if (createTask && context.recordId != null) {
            try {
                Task t = new Task();
                t.WhatId = context.recordId;
                t.Subject = config.containsKey('taskSubject')
                    ? (String) config.get('taskSubject')
                    : 'Follow up on form submission';
                Integer daysUntilDue = config.containsKey('taskDaysUntilDue')
                    ? Integer.valueOf(config.get('taskDaysUntilDue'))
                    : 3;
                t.ActivityDate = Date.today().addDays(daysUntilDue);
                t.Status = 'Not Started';
                t.Priority = 'Normal';
                t.Description = 'Auto-created by QuikForms Submission Enrichment Plugin';
                insert t;
                result.setMetadata('followUpTaskId', t.Id);
            } catch (Exception e) {
                result.addWarning('Could not create follow-up task: ' + e.getMessage());
            }
        }

        return result;
    }

    // ==================== Private Helpers ====================

    /**
     * @description Applies a case transformation to a string value
     * @param input The string to transform
     * @param transformType The transformation type: 'uppercase', 'lowercase', or 'titlecase'
     * @return The transformed string
     */
    private String applyTransform(String input, String transformType) {
        if (String.isBlank(input) || String.isBlank(transformType)) {
            return input;
        }

        switch on transformType.toLowerCase() {
            when 'uppercase' {
                return input.toUpperCase();
            }
            when 'lowercase' {
                return input.toLowerCase();
            }
            when 'titlecase' {
                return toTitleCase(input);
            }
            when else {
                return input;
            }
        }
    }

    /**
     * @description Converts a string to title case (first letter of each word capitalized)
     * @param input The string to convert
     * @return The title-cased string
     */
    private String toTitleCase(String input) {
        if (String.isBlank(input)) {
            return input;
        }

        List<String> words = input.split('\\s+');
        List<String> titleWords = new List<String>();

        for (String word : words) {
            if (String.isNotBlank(word)) {
                titleWords.add(
                    word.substring(0, 1).toUpperCase() +
                    (word.length() > 1 ? word.substring(1).toLowerCase() : '')
                );
            }
        }

        return String.join(titleWords, ' ');
    }
}
