/**
 * @description Test class for QuikFormsSubmissionEnrichmentPlugin.
 * Verifies plugin info, initialization, metadata stamping, case transformations,
 * metadata passing between handlers, and follow-up Task creation.
 *
 * @author QuikForms
 * @since 1.0
 */
@isTest
private class QuikFormsSubmissionEnrichmentPluginTest {

    // ==================== Helper Methods ====================

    /**
     * @description Creates a standard submission context for testing
     * @return QuikFormsSubmissionContext with test data populated
     */
    private static QuikFormsSubmissionContext createTestContext() {
        QuikFormsSubmissionContext ctx = new QuikFormsSubmissionContext();
        ctx.formConfigId = 'testFormConfig';
        ctx.targetObjectApiName = 'Case';
        ctx.fieldValues = new Map<String, Object>{
            'Subject' => 'Test',
            'Description' => ''
        };
        ctx.pluginConfiguration = new Map<String, Object>();
        ctx.userAgent = 'Mozilla/5.0';
        ctx.ipAddress = '192.168.1.1';
        ctx.referrer = 'https://example.com';
        ctx.metadata = new Map<String, Object>();
        return ctx;
    }

    /**
     * @description Creates and initializes a plugin with the given config
     * @param config Configuration map
     * @return Initialized plugin instance
     */
    private static QuikFormsSubmissionEnrichmentPlugin createPlugin(Map<String, Object> config) {
        QuikFormsSubmissionEnrichmentPlugin plugin = new QuikFormsSubmissionEnrichmentPlugin();
        plugin.initialize(config);
        return plugin;
    }

    // ==================== Plugin Info Tests ====================

    @isTest
    static void testGetPluginInfo() {
        QuikFormsSubmissionEnrichmentPlugin plugin = new QuikFormsSubmissionEnrichmentPlugin();

        QuikFormsPluginInfo info = plugin.getPluginInfo();

        System.assertEquals('Submission Enrichment', info.name, 'Plugin name should match');
        System.assertEquals('1.0.0', info.version, 'Version should be 1.0.0');
        System.assertNotEquals(null, info.description, 'Description should not be null');
        System.assert(info.description.length() > 0, 'Description should not be empty');
        System.assertEquals(false, info.supportsCallouts, 'Should not require callouts');
        System.assertEquals(2, info.supportedHooks.size(), 'Should support 2 hooks');
        System.assert(info.supportedHooks.contains('onBeforeSubmit'), 'Should support onBeforeSubmit');
        System.assert(info.supportedHooks.contains('onAfterSubmit'), 'Should support onAfterSubmit');
    }

    // ==================== Initialize and IsReady Tests ====================

    @isTest
    static void testInitializeAndIsReady() {
        QuikFormsSubmissionEnrichmentPlugin plugin = new QuikFormsSubmissionEnrichmentPlugin();

        System.assertEquals(false, plugin.isReady(), 'Should not be ready before initialization');

        plugin.initialize(new Map<String, Object>{ 'timestampField' => 'Description' });

        System.assertEquals(true, plugin.isReady(), 'Should be ready after initialization');
    }

    @isTest
    static void testInitializeWithNullConfig() {
        QuikFormsSubmissionEnrichmentPlugin plugin = new QuikFormsSubmissionEnrichmentPlugin();

        plugin.initialize(null);

        System.assertEquals(true, plugin.isReady(), 'Should be ready even with null config');
    }

    // ==================== onBeforeSubmit - Metadata Stamping Tests ====================

    @isTest
    static void testOnBeforeSubmitTimestampStamping() {
        Map<String, Object> config = new Map<String, Object>{
            'timestampField' => 'Description'
        };
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);
        QuikFormsSubmissionContext ctx = createTestContext();

        Datetime beforeTest = Datetime.now();
        QuikFormsSubmissionResult result = plugin.onBeforeSubmit(ctx);

        System.assertEquals(true, result.success, 'Result should be successful');
        System.assertEquals(true, result.shouldContinue, 'Should continue with submission');
        System.assert(
            result.modifiedFieldValues.containsKey('Description'),
            'Should have modified Description field'
        );
        String timestampValue = (String) result.modifiedFieldValues.get('Description');
        System.assertNotEquals(null, timestampValue, 'Timestamp value should not be null');
        System.assert(timestampValue.length() > 0, 'Timestamp value should not be empty');
    }

    @isTest
    static void testOnBeforeSubmitUserAgentStamping() {
        Map<String, Object> config = new Map<String, Object>{
            'userAgentField' => 'Description'
        };
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);
        QuikFormsSubmissionContext ctx = createTestContext();

        QuikFormsSubmissionResult result = plugin.onBeforeSubmit(ctx);

        System.assertEquals(true, result.success, 'Result should be successful');
        System.assert(
            result.modifiedFieldValues.containsKey('Description'),
            'Should have modified Description field'
        );
        System.assertEquals(
            'Mozilla/5.0',
            result.modifiedFieldValues.get('Description'),
            'User agent should match context value'
        );
    }

    @isTest
    static void testOnBeforeSubmitIpAddressStamping() {
        Map<String, Object> config = new Map<String, Object>{
            'ipAddressField' => 'Description'
        };
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);
        QuikFormsSubmissionContext ctx = createTestContext();

        QuikFormsSubmissionResult result = plugin.onBeforeSubmit(ctx);

        System.assert(
            result.modifiedFieldValues.containsKey('Description'),
            'Should have modified Description field'
        );
        System.assertEquals(
            '192.168.1.1',
            result.modifiedFieldValues.get('Description'),
            'IP address should match context value'
        );
    }

    @isTest
    static void testOnBeforeSubmitReferrerStamping() {
        Map<String, Object> config = new Map<String, Object>{
            'referrerField' => 'Description'
        };
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);
        QuikFormsSubmissionContext ctx = createTestContext();

        QuikFormsSubmissionResult result = plugin.onBeforeSubmit(ctx);

        System.assert(
            result.modifiedFieldValues.containsKey('Description'),
            'Should have modified Description field'
        );
        System.assertEquals(
            'https://example.com',
            result.modifiedFieldValues.get('Description'),
            'Referrer should match context value'
        );
    }

    @isTest
    static void testOnBeforeSubmitAllMetadataFields() {
        Map<String, Object> config = new Map<String, Object>{
            'timestampField' => 'Field_A__c',
            'userAgentField' => 'Field_B__c',
            'ipAddressField' => 'Field_C__c',
            'referrerField' => 'Field_D__c'
        };
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);
        QuikFormsSubmissionContext ctx = createTestContext();

        QuikFormsSubmissionResult result = plugin.onBeforeSubmit(ctx);

        System.assertEquals(4, result.modifiedFieldValues.size(), 'Should have 4 modified fields');
        System.assert(result.modifiedFieldValues.containsKey('Field_A__c'), 'Timestamp field should be set');
        System.assertEquals('Mozilla/5.0', result.modifiedFieldValues.get('Field_B__c'));
        System.assertEquals('192.168.1.1', result.modifiedFieldValues.get('Field_C__c'));
        System.assertEquals('https://example.com', result.modifiedFieldValues.get('Field_D__c'));
    }

    // ==================== onBeforeSubmit - Case Transform Tests ====================

    @isTest
    static void testOnBeforeSubmitCaseTransformUppercase() {
        Map<String, Object> config = new Map<String, Object>{
            'caseTransformFields' => new Map<String, Object>{
                'Subject' => 'uppercase'
            }
        };
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);
        QuikFormsSubmissionContext ctx = createTestContext();
        ctx.fieldValues.put('Subject', 'hello world');

        QuikFormsSubmissionResult result = plugin.onBeforeSubmit(ctx);

        System.assertEquals(
            'HELLO WORLD',
            result.modifiedFieldValues.get('Subject'),
            'Subject should be uppercased'
        );
    }

    @isTest
    static void testOnBeforeSubmitCaseTransformLowercase() {
        Map<String, Object> config = new Map<String, Object>{
            'caseTransformFields' => new Map<String, Object>{
                'Subject' => 'lowercase'
            }
        };
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);
        QuikFormsSubmissionContext ctx = createTestContext();
        ctx.fieldValues.put('Subject', 'HELLO WORLD');

        QuikFormsSubmissionResult result = plugin.onBeforeSubmit(ctx);

        System.assertEquals(
            'hello world',
            result.modifiedFieldValues.get('Subject'),
            'Subject should be lowercased'
        );
    }

    @isTest
    static void testOnBeforeSubmitCaseTransformTitleCase() {
        Map<String, Object> config = new Map<String, Object>{
            'caseTransformFields' => new Map<String, Object>{
                'Subject' => 'titlecase'
            }
        };
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);
        QuikFormsSubmissionContext ctx = createTestContext();
        ctx.fieldValues.put('Subject', 'hello world test');

        QuikFormsSubmissionResult result = plugin.onBeforeSubmit(ctx);

        System.assertEquals(
            'Hello World Test',
            result.modifiedFieldValues.get('Subject'),
            'Subject should be title-cased'
        );
    }

    @isTest
    static void testOnBeforeSubmitCaseTransformTitleCaseSingleWord() {
        Map<String, Object> config = new Map<String, Object>{
            'caseTransformFields' => new Map<String, Object>{
                'Subject' => 'titlecase'
            }
        };
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);
        QuikFormsSubmissionContext ctx = createTestContext();
        ctx.fieldValues.put('Subject', 'hello');

        QuikFormsSubmissionResult result = plugin.onBeforeSubmit(ctx);

        System.assertEquals(
            'Hello',
            result.modifiedFieldValues.get('Subject'),
            'Single word should be title-cased'
        );
    }

    @isTest
    static void testOnBeforeSubmitCaseTransformMultipleFields() {
        Map<String, Object> config = new Map<String, Object>{
            'caseTransformFields' => new Map<String, Object>{
                'Subject' => 'uppercase',
                'Description' => 'lowercase'
            }
        };
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);
        QuikFormsSubmissionContext ctx = createTestContext();
        ctx.fieldValues.put('Subject', 'test subject');
        ctx.fieldValues.put('Description', 'TEST DESCRIPTION');

        QuikFormsSubmissionResult result = plugin.onBeforeSubmit(ctx);

        System.assertEquals('TEST SUBJECT', result.modifiedFieldValues.get('Subject'));
        System.assertEquals('test description', result.modifiedFieldValues.get('Description'));
    }

    @isTest
    static void testOnBeforeSubmitCaseTransformNullFieldValue() {
        Map<String, Object> config = new Map<String, Object>{
            'caseTransformFields' => new Map<String, Object>{
                'MissingField' => 'uppercase'
            }
        };
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);
        QuikFormsSubmissionContext ctx = createTestContext();

        QuikFormsSubmissionResult result = plugin.onBeforeSubmit(ctx);

        System.assertEquals(true, result.success, 'Should succeed even when field is missing');
        System.assert(
            !result.modifiedFieldValues.containsKey('MissingField'),
            'Should not modify a field with null value'
        );
    }

    @isTest
    static void testOnBeforeSubmitCaseTransformUnknownType() {
        Map<String, Object> config = new Map<String, Object>{
            'caseTransformFields' => new Map<String, Object>{
                'Subject' => 'invalid_transform'
            }
        };
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);
        QuikFormsSubmissionContext ctx = createTestContext();
        ctx.fieldValues.put('Subject', 'unchanged');

        QuikFormsSubmissionResult result = plugin.onBeforeSubmit(ctx);

        System.assertEquals(
            'unchanged',
            result.modifiedFieldValues.get('Subject'),
            'Unknown transform type should return value unchanged'
        );
    }

    // ==================== onBeforeSubmit - Metadata Passing Tests ====================

    @isTest
    static void testOnBeforeSubmitMetadataPassing() {
        Map<String, Object> config = new Map<String, Object>();
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);
        QuikFormsSubmissionContext ctx = createTestContext();

        QuikFormsSubmissionResult result = plugin.onBeforeSubmit(ctx);

        System.assertEquals(
            true,
            result.metadata.get('enrichmentApplied'),
            'enrichmentApplied metadata should be true'
        );
        System.assert(
            result.metadata.containsKey('enrichmentTimestamp'),
            'enrichmentTimestamp metadata should be set'
        );
        String timestamp = (String) result.metadata.get('enrichmentTimestamp');
        System.assertNotEquals(null, timestamp, 'enrichmentTimestamp should not be null');
        System.assert(timestamp.length() > 0, 'enrichmentTimestamp should not be empty');
    }

    // ==================== onBeforeSubmit - No Config Tests ====================

    @isTest
    static void testOnBeforeSubmitNoConfig() {
        Map<String, Object> config = new Map<String, Object>();
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);
        QuikFormsSubmissionContext ctx = createTestContext();

        QuikFormsSubmissionResult result = plugin.onBeforeSubmit(ctx);

        System.assertEquals(true, result.success, 'Should succeed with empty config');
        System.assertEquals(true, result.shouldContinue, 'Should continue with empty config');
        System.assertEquals(0, result.errorMessages.size(), 'Should have no errors');
        // Only metadata keys should be present (enrichmentApplied, enrichmentTimestamp)
        // No modified field values from stamping or transforms
        System.assert(
            result.metadata.containsKey('enrichmentApplied'),
            'Metadata should still be set even with empty config'
        );
    }

    @isTest
    static void testOnBeforeSubmitBlankFieldNames() {
        Map<String, Object> config = new Map<String, Object>{
            'timestampField' => '',
            'userAgentField' => '  ',
            'ipAddressField' => null,
            'referrerField' => ''
        };
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);
        QuikFormsSubmissionContext ctx = createTestContext();

        QuikFormsSubmissionResult result = plugin.onBeforeSubmit(ctx);

        System.assertEquals(true, result.success, 'Should succeed with blank field names');
        // ipAddressField is null which would cause a NullPointerException on isNotBlank
        // but config.get('ipAddressField') returns null, and we cast to String which is ok
        // Only metadata keys should exist, no field modifications from blank field names
    }

    // ==================== onAfterSubmit - Task Creation Tests ====================

    @isTest
    static void testOnAfterSubmitCreateTask() {
        Map<String, Object> config = new Map<String, Object>{
            'createFollowUpTask' => true,
            'taskSubject' => 'Review form submission',
            'taskDaysUntilDue' => 5
        };
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);

        // Create a real Case record to link the Task to
        Case testCase = new Case(Subject = 'Test Case', Status = 'New');
        insert testCase;

        QuikFormsSubmissionContext ctx = createTestContext();
        ctx.recordId = testCase.Id;

        QuikFormsSubmissionResult result = plugin.onAfterSubmit(ctx);

        System.assertEquals(true, result.success, 'Result should be successful');
        System.assert(
            result.metadata.containsKey('followUpTaskId'),
            'Should have followUpTaskId in metadata'
        );

        // Verify the Task was actually created
        Id taskId = (Id) result.metadata.get('followUpTaskId');
        Task createdTask = [
            SELECT Id, Subject, WhatId, ActivityDate, Status, Priority, Description
            FROM Task
            WHERE Id = :taskId
        ];
        System.assertEquals('Review form submission', createdTask.Subject);
        System.assertEquals(testCase.Id, createdTask.WhatId);
        System.assertEquals(Date.today().addDays(5), createdTask.ActivityDate);
        System.assertEquals('Not Started', createdTask.Status);
        System.assertEquals('Normal', createdTask.Priority);
        System.assert(
            createdTask.Description.contains('QuikForms Submission Enrichment Plugin'),
            'Task description should reference the plugin'
        );
    }

    @isTest
    static void testOnAfterSubmitCreateTaskDefaultValues() {
        Map<String, Object> config = new Map<String, Object>{
            'createFollowUpTask' => true
        };
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);

        Case testCase = new Case(Subject = 'Test Case Defaults', Status = 'New');
        insert testCase;

        QuikFormsSubmissionContext ctx = createTestContext();
        ctx.recordId = testCase.Id;

        QuikFormsSubmissionResult result = plugin.onAfterSubmit(ctx);

        System.assertEquals(true, result.success, 'Result should be successful');
        System.assert(
            result.metadata.containsKey('followUpTaskId'),
            'Should have followUpTaskId in metadata'
        );

        Id taskId = (Id) result.metadata.get('followUpTaskId');
        Task createdTask = [
            SELECT Id, Subject, ActivityDate
            FROM Task
            WHERE Id = :taskId
        ];
        System.assertEquals('Follow up on form submission', createdTask.Subject, 'Default subject should be used');
        System.assertEquals(Date.today().addDays(3), createdTask.ActivityDate, 'Default 3 days should be used');
    }

    @isTest
    static void testOnAfterSubmitNoTask() {
        Map<String, Object> config = new Map<String, Object>{
            'createFollowUpTask' => false
        };
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);

        QuikFormsSubmissionContext ctx = createTestContext();
        ctx.recordId = QuikFormsTestIdGenerator.getFakeId(Case.SObjectType);

        Integer taskCountBefore = [SELECT COUNT() FROM Task];

        QuikFormsSubmissionResult result = plugin.onAfterSubmit(ctx);

        Integer taskCountAfter = [SELECT COUNT() FROM Task];
        System.assertEquals(true, result.success, 'Result should be successful');
        System.assert(
            !result.metadata.containsKey('followUpTaskId'),
            'Should not have followUpTaskId when task creation is disabled'
        );
        System.assertEquals(taskCountBefore, taskCountAfter, 'No new Tasks should be created');
    }

    @isTest
    static void testOnAfterSubmitNoTaskWhenNotConfigured() {
        Map<String, Object> config = new Map<String, Object>();
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);

        QuikFormsSubmissionContext ctx = createTestContext();
        ctx.recordId = QuikFormsTestIdGenerator.getFakeId(Case.SObjectType);

        Integer taskCountBefore = [SELECT COUNT() FROM Task];

        QuikFormsSubmissionResult result = plugin.onAfterSubmit(ctx);

        Integer taskCountAfter = [SELECT COUNT() FROM Task];
        System.assertEquals(true, result.success, 'Result should be successful');
        System.assertEquals(taskCountBefore, taskCountAfter, 'No new Tasks should be created');
    }

    @isTest
    static void testOnAfterSubmitNoTaskWhenNullRecordId() {
        Map<String, Object> config = new Map<String, Object>{
            'createFollowUpTask' => true
        };
        QuikFormsSubmissionEnrichmentPlugin plugin = createPlugin(config);

        QuikFormsSubmissionContext ctx = createTestContext();
        ctx.recordId = null;

        Integer taskCountBefore = [SELECT COUNT() FROM Task];

        QuikFormsSubmissionResult result = plugin.onAfterSubmit(ctx);

        Integer taskCountAfter = [SELECT COUNT() FROM Task];
        System.assertEquals(true, result.success, 'Result should be successful');
        System.assertEquals(taskCountBefore, taskCountAfter, 'No Task should be created when recordId is null');
    }
}
